unit WebExam1;

{
  Display an exam on the web, administer and score it.
  Requires MSXML v3 package from Microsoft.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Version 1.0 - 22 October, 1999.
}

interface

uses
  Windows, Messages, SysUtils, Classes, HTTPApp, Exams, ActiveX, MSXML2_TLB,
  Dialogs;

type
  TwmdExam = class(TWebModule)
    procedure wmbExamwacExamAction(Sender: TObject; Request: TWebRequest;
      Response: TWebResponse; var Handled: Boolean);
  private
  public
  end;

var
  wmdExam: TwmdExam;

implementation

{$R *.DFM}

const
  { Command strings }
  GotoAction     = 'Goto';
  NextAction     = 'Next';
  PreviousAction = 'Previous';
  ResultsAction  = 'Results';
  { HTML fields }
  ActionField   = 'act';
  AnswerField   = 'answer';
  ExamField     = 'exam';
  GetMethod     = 'GET';
  QuestionField = 'qno';
  SessionField  = 'ses';
  { XSLT parameters }
  AnswerParam     = 'answer';
  AnswersParam    = 'answers';
  EngineParam     = 'engine';
  QuestionIdParam = 'qid';
  QuestionParam   = 'qno';
  ResultsParam    = 'results';
  SessionParam    = 'ses';
  { Source files for transformations }
  ExamIntro    = 'ExamIntro.xsl';
  ExamQuestion = 'ExamQuestion.xsl';
  ExamResults  = 'ExamResults.xsl';
  { Range of session ids }
  SessionRange = 32000;

var
  BaseDir: string;
  ExamList: TStringList;
  Index: Integer;
  Sessions: TStringList;
  XMLDocs: TInterfaceList;
  XSLIntro: IXMLDOMDocument;
  XSLIntroTemplate: IXSLTemplate;
  XSLQuestion: IXMLDOMDocument;
  XSLQuestionTemplate: IXSLTemplate;
  XSLResults: IXMLDOMDocument;
  XSLResultsTemplate: IXSLTemplate;

{ TwmdExam --------------------------------------------------------------------}

{ Generate the appropriate HTML page from the exam and user responses }
procedure TwmdExam.wmbExamwacExamAction(Sender: TObject;
  Request: TWebRequest; Response: TWebResponse; var Handled: Boolean);
var
  CurQuestion: Integer;
  Exam: TExam;
  Fields: TStrings;
  QuestionId: string;
  ScriptName: string;
  Session: TUserSession;
  SessionId: string;
  UserAction: string;
  XMLDoc: IXMLDOMDocument;
  XSLProcessor: IXSLProcessor;

  { Save any answers from the current page }
  procedure ProcessAnswers(QuestionId: string);
  var
    Answer: string;
    Index: Integer;
    Question: TQuestion;
  begin
    { If already answered then can't change it }
    if Session.IsAnswered[QuestionId] then
      Exit;

    { Retrieve answer(s) based on type }
    Question := Session.Exam.QuestionById[QuestionId];
    if Question.AnswerType = atCheckbox then
    begin
      Answer := '';
      { Could be multiple selections }
      for Index := 0 to Fields.Count - 1 do
        if Fields.Names[Index] = AnswerField then
          Answer :=
            Answer + ',' + Copy(Fields[Index], Length(AnswerField) + 2, 255);
      if Answer <> '' then
        Delete(Answer, 1, 1);
    end
    else
      Answer := Fields.Values[AnswerField];
    Session.Answer[QuestionId] := Answer;
  end;

  { Initialisation - extract common fields and save any answer }
  procedure Initialise;
  var
    ExamName: string;
    Index: Integer;
  begin
    { Extract the script location }
    ScriptName := Request.ScriptName;
    { Set up a pointer to the request fields }
    if Request.Method = GetMethod then
      Fields   := Request.QueryFields
    else
      Fields   := Request.ContentFields;
    { Retrieve the current session details }
    SessionId  := Fields.Values[SessionField];
    Index      := Sessions.IndexOf(SessionId);
    if Index > -1 then
    begin
      { Retrieve previous session details }
      Session := TUserSession(Sessions.Objects[Index]);
      Exam    := Session.Exam;
      XMLDoc  := XMLDocs[ExamList.IndexOfObject(Exam)] as IXMLDOMDocument;
    end
    else
    begin
      ExamName := Fields.Values[ExamField];
      Index    := ExamList.IndexOf(ExamName);
      if Index > -1 then
      begin
        { Retrieve existing exam }
        Exam   := TExam(ExamList.Objects[Index]);
        XMLDoc := XMLDocs[Index] as IXMLDOMDocument;
      end
      else
      begin
        { Load new exam }
        Exam   := LoadExam(BaseDir + ExamName);
        { Create XML object model for exam }
        XMLDoc := CoDOMDocument.Create;
        if not XMLDoc.Load(BaseDir + ExamName) then
          raise Exception.Create('Couldn''t load exam doc'#13 +
            XMLDoc.ParseError.Reason);
        { Cache for future reference }
        ExamList.AddObject(ExamName, Exam);
        XMLDocs.Add(XMLDoc);
      end;
      { Create a new user session }
      Session := TUserSession.Create(Exam);
      repeat
        SessionId := IntToStr(Random(SessionRange));
      until Sessions.IndexOf(SessionId) = -1;
      Sessions.AddObject(SessionId, Session);
    end;
    { Extract the user requested action }
    UserAction := Fields.Values[ActionField];
    try
      { Get the current question number }
      CurQuestion := StrToInt(Fields.Values[QuestionField]);
      QuestionId  := Session.QuestionId[CurQuestion - 1];
      { Do we need to save any current answers? }
      if UserAction <> GotoAction then
        ProcessAnswers(QuestionId);
    except
      { Ignore }
    end;
  end;

  { Has the exam been completed? }
  function Completed: Boolean;
  begin
    Result := (Session.Answered = Session.QuestionCount);
  end;

  { Compile a string representing the results of answering the questions -
    'Y' for correct, 'N' for incorrect, ' ' for unanswered }
  function CompileAnswers: string;
  const
    Correct: array [Boolean] of string = ('N', 'Y');
  var
    Index: Integer;
    QuestionId: string;
  begin
    for Index := 0 to Session.QuestionCount - 1 do
    begin
      QuestionId := Session.QuestionId[Index];
      if Session.IsAnswered[QuestionId] then
        Result := Result + Correct[
          Exam.QuestionById[QuestionId].IsValid(Session.Answer[QuestionId])]
      else
        Result := Result + ' ';
    end;
  end;

  { Create an XSL processor and set common properties }
  function PrepareProcessor(XSLTemplate: IXSLTemplate): IXSLProcessor;
  begin
    Result       := XSLTemplate.CreateProcessor;
    Result.AddParameter(EngineParam, ScriptName, '');
    Result.AddParameter(SessionParam, SessionId, '');
    Result.Input := XMLDoc;
  end;

begin
  try
    { Extract common fields and save any answer }
    Initialise;

    { Move to a new question }
    if UserAction = NextAction then
      Inc(CurQuestion)
    else if UserAction = PreviousAction then
    begin
      Dec(CurQuestion);
      if CurQuestion < 1 then
        CurQuestion := 1;
    end;
    { Perform the requested action }
    if (UserAction = ResultsAction) or (CurQuestion > Session.QuestionCount) then
    begin
      { Show the scoring page }
      XSLProcessor     := PrepareProcessor(XSLResultsTemplate);
      XSLProcessor.AddParameter(AnswersParam, CompileAnswers, '');
      XSLProcessor.Transform;
      Response.Content := XSLProcessor.Output;
    end
    else if (UserAction = NextAction) or (UserAction = PreviousAction) or
      (UserAction = GotoAction) then
    begin
      { Show a question page }
      QuestionId       := Session.QuestionId[CurQuestion - 1];
      XSLProcessor     := PrepareProcessor(XSLQuestionTemplate);
      XSLProcessor.AddParameter(AnswerParam, Session.Answer[QuestionId], '');
      XSLProcessor.AddParameter(QuestionIdParam, QuestionId, '');
      XSLProcessor.AddParameter(QuestionParam, CurQuestion, '');
      XSLProcessor.AddParameter(ResultsParam, Completed, '');
      XSLProcessor.Transform;
      Response.Content := XSLProcessor.Output;
    end
    else
    begin
      { Default - show the introduction }
      XSLProcessor     := PrepareProcessor(XSLIntroTemplate);
      XSLProcessor.Transform;
      Response.Content := XSLProcessor.Output;
    end;
  finally
    Response.ContentType := 'text/html';
    Handled              := True;
    XMLDoc               := nil;
    XSLProcessor         := nil;
  end;
end;

initialization
  BaseDir  := 'c:\inetpub\scripts\';
  { Create lists for exams and user sessions }
  ExamList := TStringList.Create;
  Sessions := TStringList.Create;
  XMLDocs  := TInterfaceList.Create;
  { Create template for introduction stylesheet }
  XSLIntro                       := CoFreeThreadedDOMDocument.Create;
  if not XSLIntro.Load(BaseDir + ExamIntro) then
    MessageDlg('Couldn''t load exam intro'#13 + XSLIntro.ParseError.Reason,
      mtError, [mbOK], 0);
  XSLIntroTemplate               := CoXSLTemplate.Create;
  XSLIntroTemplate.Stylesheet    := XSLIntro;
  { Create template for individual question stylesheet }
  XSLQuestion                    := CoFreeThreadedDOMDocument.Create;
  if not XSLQuestion.Load(BaseDir + ExamQuestion) then
    MessageDlg('Couldn''t load exam question'#13 +
      XSLQuestion.ParseError.Reason, mtError, [mbOK], 0);
  XSLQuestionTemplate            := CoXSLTemplate.Create;
  XSLQuestionTemplate.Stylesheet := XSLQuestion;
  { Create template for results stylesheet }
  XSLResults                     := CoFreeThreadedDOMDocument.Create;
  if not XSLResults.Load(BaseDir + ExamResults) then
    MessageDlg('Couldn''t load exam results'#13 + XSLResults.ParseError.Reason,
      mtError, [mbOK], 0);
  XSLResultsTemplate             := CoXSLTemplate.Create;
  XSLResultsTemplate.Stylesheet  := XSLResults;
finalization
  { Release resources }
  for Index := 0 to ExamList.Count - 1 do
    ExamList.Objects[Index].Free;
  ExamList.Free;
  for Index := 0 to Sessions.Count - 1 do
    Sessions.Objects[Index].Free;
  Sessions.Free;
  XMLDocs.Free;
  XSLIntroTemplate    := nil;
  XSLIntro            := nil;
  XSLQuestionTemplate := nil;
  XSLQuestion         := nil;
  XSLResultsTemplate  := nil;
  XSLResults          := nil;
end.
